import { useState, useEffect } from 'react'
import { motion } from 'framer-motion'
import { Video, User, Award, Play, Plus, TrendingUp, Clock, Zap } from 'lucide-react'
import { useNavigate } from 'react-router-dom'
import ImageSlider from '../components/ImageSlider'
import './MyProjects.css'

const MyProjects = () => {
  const navigate = useNavigate()
  const [projects, setProjects] = useState([])

  useEffect(() => {
    const saved = localStorage.getItem('videoProjects')
    if (saved) {
      try {
        setProjects(JSON.parse(saved))
      } catch (e) {
        setProjects([])
      }
    } else {
      const defaultProjects = [
        {
          id: 1,
          course: 'Premiere Pro: Продвинутый уровень',
          instructor: 'Алексей Кинематографов',
          work: 'Короткометражный фильм "Городские ритмы"',
          rating: 5,
          photo: '/images/project-1.jpg'
        },
        {
          id: 2,
          course: 'DaVinci Resolve: Цветокоррекция',
          instructor: 'Мария Режиссерова',
          work: 'Музыкальный клип "Неоновые ночи"',
          rating: 4,
          photo: '/images/project-2.jpg'
        },
        {
          id: 3,
          course: 'Final Cut Pro: Монтаж для YouTube',
          instructor: 'Дмитрий Видеографов',
          work: 'Влог-серия "Путешествия"',
          rating: 5,
          photo: '/images/project-3.jpg'
        }
      ]
      setProjects(defaultProjects)
      localStorage.setItem('videoProjects', JSON.stringify(defaultProjects))
    }
  }, [])

  return (
    <div className="my-projects-tech">
      <div className="container">
        <motion.section 
          className="hero-tech-section"
          initial={{ opacity: 0 }}
          animate={{ opacity: 1 }}
          transition={{ duration: 0.8 }}
        >
          <div className="hero-content-tech">
            <motion.div 
              className="hero-badge-tech"
              initial={{ scale: 0 }}
              animate={{ scale: 1 }}
              transition={{ delay: 0.2, type: "spring" }}
            >
              <Video size={16} strokeWidth={2.5} />
              <span>МОИ ПРОЕКТЫ</span>
            </motion.div>
            
            <motion.h1 
              className="hero-title-tech"
              initial={{ y: 30, opacity: 0 }}
              animate={{ y: 0, opacity: 1 }}
              transition={{ delay: 0.3 }}
            >
              Мои <span className="title-gradient-tech">проекты</span>
            </motion.h1>
            
            <motion.p 
              className="hero-description-tech"
              initial={{ y: 20, opacity: 0 }}
              animate={{ y: 0, opacity: 1 }}
              transition={{ delay: 0.4 }}
            >
              Отслеживайте свои работы, оценки преподавателей и продолжайте развиваться в видеомонтаже
            </motion.p>
            
            <motion.button 
              className="cta-tech-button"
              onClick={() => navigate('/enroll')}
              initial={{ y: 20, opacity: 0 }}
              animate={{ y: 0, opacity: 1 }}
              transition={{ delay: 0.5 }}
              whileHover={{ scale: 1.05 }}
              whileTap={{ scale: 0.95 }}
            >
              <Plus size={20} strokeWidth={2.5} />
              <span>Новый курс</span>
            </motion.button>
          </div>
          
          <motion.div 
            className="hero-visual-tech"
            initial={{ x: 100, opacity: 0 }}
            animate={{ x: 0, opacity: 1 }}
            transition={{ delay: 0.6 }}
          >
            <div className="tech-visualizer">
              <div className="tech-ring ring-1"></div>
              <div className="tech-ring ring-2"></div>
              <div className="tech-ring ring-3"></div>
              <div className="tech-center">
                <Video size={48} strokeWidth={2} />
              </div>
            </div>
          </motion.div>
        </motion.section>

        <ImageSlider />

        <section className="projects-stack-section">
          {projects.length === 0 ? (
            <motion.div 
              className="empty-state-tech"
              initial={{ opacity: 0, scale: 0.9 }}
              animate={{ opacity: 1, scale: 1 }}
            >
              <div className="empty-icon-container-tech">
                <Video size={64} strokeWidth={1.5} />
                <div className="empty-pulse-tech"></div>
              </div>
              <h2 className="empty-title-tech">Нет проектов</h2>
              <p className="empty-text-tech">Начните с записи на первый курс</p>
              <button className="empty-button-tech" onClick={() => navigate('/enroll')}>
                Записаться
              </button>
            </motion.div>
          ) : (
            <div className="projects-stack-tech">
              {projects.map((project, index) => {
                return (
                  <motion.div
                    key={project.id}
                    className="project-card-tech"
                    initial={{ opacity: 0, y: 50 }}
                    animate={{ opacity: 1, y: 0 }}
                    transition={{ delay: 0.2 + index * 0.15 }}
                    whileHover={{ y: -10 }}
                    style={{ zIndex: projects.length - index }}
                  >
                    <div className="project-image-tech">
                      <img src={project.photo} alt={project.work} />
                      <div className="project-overlay-tech">
                        <button className="play-button-tech">
                          <Play size={28} strokeWidth={2.5} fill="currentColor" />
                        </button>
                      </div>
                      <div className="project-rating-tech">
                        <Award size={16} strokeWidth={2} />
                        <span>{project.rating}/5</span>
                      </div>
                      <div className="project-gradient-tech"></div>
                    </div>
                    
                    <div className="project-info-tech">
                      <h3 className="project-title-tech">{project.course}</h3>
                      <div className="project-meta-tech">
                        <div className="meta-item-tech">
                          <User size={16} strokeWidth={2} />
                          <span>{project.instructor}</span>
                        </div>
                        <div className="meta-item-tech">
                          <Video size={16} strokeWidth={2} />
                          <span>{project.work}</span>
                        </div>
                      </div>
                    </div>
                  </motion.div>
                )
              })}
            </div>
          )}
        </section>

        <section className="stats-tech-section">
          <motion.div 
            className="stats-container-tech"
            initial={{ opacity: 0 }}
            whileInView={{ opacity: 1 }}
            viewport={{ once: true }}
            transition={{ duration: 0.6 }}
          >
            <div className="stat-item-tech">
              <div className="stat-icon-tech">
                <Video size={24} strokeWidth={2} />
              </div>
              <div className="stat-value-tech">{projects.length}</div>
              <div className="stat-label-tech">Проектов</div>
            </div>
            <div className="stat-item-tech">
              <div className="stat-icon-tech">
                <TrendingUp size={24} strokeWidth={2} />
              </div>
              <div className="stat-value-tech">
                {projects.length > 0 
                  ? (projects.reduce((sum, p) => sum + (p.rating || 0), 0) / projects.length).toFixed(1)
                  : '0.0'}
              </div>
              <div className="stat-label-tech">Средняя оценка</div>
            </div>
            <div className="stat-item-tech">
              <div className="stat-icon-tech">
                <Clock size={24} strokeWidth={2} />
              </div>
              <div className="stat-value-tech">24/7</div>
              <div className="stat-label-tech">Поддержка</div>
            </div>
            <div className="stat-item-tech">
              <div className="stat-icon-tech">
                <Zap size={24} strokeWidth={2} />
              </div>
              <div className="stat-value-tech">100%</div>
              <div className="stat-label-tech">Практика</div>
            </div>
          </motion.div>
        </section>
      </div>
    </div>
  )
}

export default MyProjects

